const db = require('../models');
const { Op } = require('sequelize');

exports.getRESTAFFWithStats = async (userId) => {
  const where = { app: 'RESTAFF' };
  if (userId) where.user_id = userId;
  const clients = await db.Client.findAll({
    where,
    raw: true
  });

  if (clients.length === 0) return [];

  const refids = [...new Set(clients.map(c => c.refid).filter(Boolean))];
  if (refids.length === 0) {
    return clients.map(c => ({ ...c, branch_name: '-', total: 0, remain: 0, off: 0, nr: 0, pr: 0, mc: 0, rcv: 0 }));
  }

  const [statsRows, branchRows] = await Promise.all([
    db.Import.findAll({
      attributes: [
        'refid',
        [db.sequelize.literal(`SUM(CASE WHEN code LIKE 'C%' THEN 1 ELSE 0 END)`), 'total'],
        [db.sequelize.literal(`SUM(CASE WHEN code LIKE 'C%' AND (remarks = '' OR remarks IS NULL) THEN 1 ELSE 0 END)`), 'remain'],
        [db.sequelize.literal(`SUM(CASE WHEN code LIKE 'C%' AND LEFT(remarks, 2) = 'of' THEN 1 ELSE 0 END)`), 'off'],
        [db.sequelize.literal(`SUM(CASE WHEN code LIKE 'C%' AND LEFT(remarks, 2) = 'nr' THEN 1 ELSE 0 END)`), 'nr'],
        [db.sequelize.literal(`SUM(CASE WHEN code LIKE 'C%' AND LEFT(remarks, 2) = 'pr' THEN 1 ELSE 0 END)`), 'pr'],
        [db.sequelize.literal(`SUM(CASE WHEN LEFT(remarks, 2) = 'mc' THEN 1 ELSE 0 END)`), 'mc'],
        [db.sequelize.literal(`SUM(CASE WHEN LEFT(remarks, 2) = 'rc' THEN 1 ELSE 0 END)`), 'rcv']
      ],
      where: { refid: { [Op.in]: refids } },
      group: ['refid'],
      raw: true
    }),
    db.TblSave.findAll({
      attributes: ['refid', 'name'],
      where: { refid: { [Op.in]: refids } },
      order: [['id', 'DESC']],
      raw: true
    })
  ]);

  const stats = {};
  statsRows.forEach(r => { stats[r.refid] = r; });
  const branchNames = {};
  branchRows.forEach(r => { if (!branchNames[r.refid]) branchNames[r.refid] = r.name; });

  return clients.map(c => ({
    ...c,
    branch_name: branchNames[c.refid] || '-',
    total: stats[c.refid]?.total || 0,
    remain: stats[c.refid]?.remain || 0,
    off: stats[c.refid]?.off || 0,
    nr: stats[c.refid]?.nr || 0,
    pr: stats[c.refid]?.pr || 0,
    mc: stats[c.refid]?.mc || 0,
    rcv: stats[c.refid]?.rcv || 0
  }));
};

exports.getCRCWithBranch = async (userId) => {
  const where = { app: 'CRC' };
  if (userId) where.user_id = userId;
  const clients = await db.Client.findAll({
    where,
    raw: true
  });

  if (clients.length === 0) return [];

  const refids = [...new Set(clients.map(c => c.refid).filter(Boolean))];
  if (refids.length === 0) {
    return clients.map(c => ({ ...c, branch_name: '-' }));
  }

  const branchRows = await db.TblSaveAudit.findAll({
    attributes: ['refid', 'name'],
    where: { refid: { [Op.in]: refids } },
    order: [['id', 'DESC']],
    raw: true
  });

  const branchNames = {};
  branchRows.forEach(r => { if (!branchNames[r.refid]) branchNames[r.refid] = r.name; });

  return clients.map(c => ({
    ...c,
    branch_name: branchNames[c.refid] || '-'
  }));
};
