const multer = require('multer');
const path = require('path');
const config = require('../config');
const db = require('../models');
const excelService = require('../services/excelService');

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, config.uploadExcelPath);
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + (file.originalname || 'upload.xlsx'));
  }
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname || '').toLowerCase();
    if (['.xlsx', '.xls'].includes(ext)) {
      cb(null, true);
    } else {
      cb(new Error('Only Excel files (.xlsx, .xls) allowed'));
    }
  }
}).single('userfile');

const handleUpload = (req, res, next) => {
  upload(req, res, (err) => {
    if (err) {
      return res.status(400).json({ success: false, message: err.message });
    }
    next();
  });
};

exports.index = async (req, res) => {
  res.json({ message: 'Upload form' });
};

exports.branches = async (req, res) => {
  try {
    const branches = await db.TblSave.findAll({
      attributes: ['name', 'refid'],
      order: [['id', 'DESC']],
      raw: true
    });
    res.json({ success: true, branches });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch branches' });
  }
};

exports.display = async (req, res) => {
  try {
    const { refid, page = 1, per_page = 150 } = req.query;
    const where = {};
    if (refid) where.refid = refid;

    const { count, rows } = await db.Import.findAndCountAll({
      where,
      limit: parseInt(per_page),
      offset: (parseInt(page) - 1) * parseInt(per_page),
      order: [['id', 'DESC']],
      raw: true
    });

    const branches = await db.TblSave.findAll({
      attributes: ['name', 'refid'],
      order: [['id', 'DESC']],
      raw: true
    });

    res.json({
      success: true,
      sheetData: rows,
      branches,
      total_rows: count,
      page: parseInt(page),
      per_page: parseInt(per_page)
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch data' });
  }
};

exports.save = [
  handleUpload,
  async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ success: false, message: 'No file uploaded' });
      }
      const name = req.body.name || 'Import';
      const result = await excelService.importRESTAFF(req.file.path, req.userId, name);
      res.json({ success: true, ...result });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: err.message || 'Import failed' });
    }
  }
];

exports.saveRange = [
  handleUpload,
  async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ success: false, message: 'No file uploaded' });
      }
      const { name, header_name, client_id, branch_id, start, end } = req.body;
      const result = await excelService.importRESTAFF(req.file.path, req.userId, name || header_name, {
        startRow: parseInt(start) || 2,
        endRow: parseInt(end) || 9999,
        fixedHeaderName: header_name || null
      });

      if (client_id && branch_id) {
        await db.Client.update(
          { branch_id, refid: result.refid },
          { where: { id: client_id, user_id: req.userId } }
        );
      }
      res.json({ success: true, ...result });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: err.message || 'Import failed' });
    }
  }
];

exports.createExcel = async (req, res) => {
  try {
    const { refid, branches, fileName } = req.body;
    const buffer = await excelService.exportRESTAFF(refid || null, branches || null);

    let downloadName = fileName || 'export';
    if (refid) {
      const row = await db.TblSave.findOne({ where: { refid }, attributes: ['name'] });
      if (row) downloadName = row.name;
    }
    const filename = `${downloadName}_${new Date().toISOString().slice(0, 19).replace(/[:T]/g, '-')}.xlsx`;

    res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
    res.send(buffer);
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Export failed' });
  }
};

exports.delBranch = async (req, res) => {
  try {
    const { refid } = req.body;
    if (!refid) {
      return res.status(400).json({ success: false, message: 'refid required' });
    }
    await db.Import.destroy({ where: { refid } });
    await db.TblSave.destroy({ where: { refid } });
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Delete failed' });
  }
};
