const multer = require('multer');
const path = require('path');
const bcrypt = require('bcrypt');
const db = require('../models');
const config = require('../config');

const out = (source, status, title, message) => ({
  status,
  title,
  message,
  source: source != null ? JSON.stringify(source) : ' null'
});

const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, config.uploadPath),
  filename: (req, file, cb) => cb(null, Date.now() + '-' + (file.originalname || 'audio'))
});

const uploadAudio = multer({ storage }).single('uploadedfile');

exports.restaffLogin = async (req, res) => {
  try {
    const { username, password } = req.body;
    if (!username || !password) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }

    const client = await db.Client.findOne({
      where: { ename: username, app: 'RESTAFF' },
      include: [{ model: db.Branch, as: 'Branch', attributes: ['name'] }]
    });

    if (!client) {
      return res.json(out(null, false, 'Login Error', 'Incorrect credentials'));
    }

    const valid = await bcrypt.compare(password, client.epassword).catch(() => false);
    if (!valid && client.epassword !== password) {
      return res.json(out(null, false, 'Login Error', 'Incorrect credentials'));
    }

    const lists = await db.Import.findAll({
      where: { refid: client.refid },
      raw: true
    });
    const list = lists.filter(l => !l.remarks || l.remarks === '');

    const messages = await db.Message.findOne({
      where: { branch_id: client.branch_id },
      raw: true
    });

    const data = {
      id: client.id,
      branch_name: client.Branch?.name || '',
      c1_message: messages?.c1_message || '',
      c2_message: messages?.c2_message || '',
      g1_message: messages?.g1_message || '',
      g2_message: messages?.g2_message || '',
      list
    };
    res.json(out(data, true, 'Login', ''));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};

exports.restaffSubmitRecord = async (req, res) => {
  try {
    const { remarks, path: pathVal, user_id, refid, call_time } = req.body;
    if (!remarks || !pathVal || !user_id || !refid) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }

    let pathToSave = 'true';
    if (pathVal !== 'true') {
      const parts = pathVal.split('/');
      pathToSave = parts[parts.length - 1] || 'true';
    }

    const updateData = { remarks, path: pathToSave };
    if (call_time) updateData.call_time = call_time;

    await db.Import.update(updateData, { where: { id: refid } });
    res.json(out('true', true, 'Success-Record', 'Data submit successfully'));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};

exports.restaffAudioFile = (req, res) => {
  uploadAudio(req, res, (err) => {
    if (err) return res.send('fail');
    res.send('success');
  });
};

exports.restaffUpdateStatus = async (req, res) => {
  try {
    const { user_status, user_id } = req.body;
    if (!user_status || !user_id) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }
    await db.Client.update({ status: user_status }, { where: { id: user_id } });
    res.json(out(null, true, 'Status', ''));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};

exports.crcLogin = async (req, res) => {
  try {
    const { username, password } = req.body;
    if (!username || !password) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }

    const client = await db.Client.findOne({
      where: { ename: username, app: 'CRC' },
      include: [{ model: db.Branch, as: 'Branch', attributes: ['name'] }]
    });

    if (!client) {
      return res.json(out(null, false, 'Login Error', 'No access'));
    }

    const valid = await bcrypt.compare(password, client.epassword).catch(() => false);
    if (!valid && client.epassword !== password) {
      return res.json(out(null, false, 'Login Error', 'No access'));
    }

    const data = {
      id: client.id,
      name: client.ename,
      phone: client.ephone,
      branch_name: client.Branch?.name || '',
      app: client.app,
      refid: client.refid,
      status: client.status
    };
    res.json(out(data, true, 'Login', 'Successfully logged in'));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};

exports.crcSubmitRecord = async (req, res) => {
  try {
    const body = req.body;
    const required = ['acc_no', 'home_verified', 'inquiry_name', 'rental', 'repute', 'worth', 'product_verified', 'remarks', 'client_id', 'refid', 'phone_status', 'branch_name'];
    const missing = required.filter(k => !body[k]);
    if (missing.length) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }

    const now = new Date().toLocaleString('en-US', { hour12: true });
    const logData = {
      k: body.acc_no,
      l: body.home_verified,
      m: body.rental,
      n: body.repute,
      o: body.worth,
      p: body.product_verified,
      s: body.remarks,
      t: body.inquiry_name,
      client_id: body.client_id,
      refid: body.refid,
      w: body.phone_status,
      x: body.branch_name,
      attempt_datetime: now
    };
    await db.TblCrcLog.create(logData);

    const existing = await db.TblAuditExcel.findOne({
      where: { b: body.acc_no, refid: body.refid }
    });

    const updateData = {
      l: body.home_verified,
      m: body.rental,
      n: body.repute,
      o: body.worth,
      p: body.product_verified,
      s: body.remarks,
      w: body.phone_status,
      x: body.branch_name,
      attempt_datetime: now,
      client_id: body.client_id
    };

    if (existing) {
      await existing.update(updateData);
    } else {
      await db.TblAuditExcel.create({
        b: body.acc_no,
        refid: body.refid,
        ...updateData
      });
    }

    res.json(out(null, true, 'Success-Record', 'Data submit successfully'));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};

exports.crcUpdateStatus = async (req, res) => {
  try {
    const { user_status, user_id } = req.body;
    if (!user_status || !user_id) {
      return res.json(out(null, false, 'Error', 'Invalid parameters'));
    }
    await db.Client.update({ status: user_status }, { where: { id: user_id } });
    res.json(out(null, true, 'Status', 'Status updated'));
  } catch (err) {
    console.error(err);
    res.json(out(null, false, 'Error', 'Server error'));
  }
};
