const { body, validationResult } = require('express-validator');
const db = require('../models');

exports.list = async (req, res) => {
  try {
    const [messages, branches] = await Promise.all([
      db.Message.findAll({
        where: { user_id: req.userId },
        include: [{ model: db.Branch, as: 'Branch', attributes: ['id', 'name'] }],
        order: [['id', 'DESC']]
      }),
      db.Branch.findAll({ where: { user_id: req.userId }, raw: true })
    ]);
    res.json({ success: true, messages, branches });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch messages' });
  }
};

exports.getMessage = async (req, res) => {
  try {
    const message = await db.Message.findByPk(req.params.id);
    if (!message) return res.status(404).json({ success: false, message: 'Message not found' });
    if (message.user_id !== req.userId) {
      return res.status(403).json({ success: false, message: 'Forbidden' });
    }
    res.json({ success: true, message: message.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch message' });
  }
};

exports.add = [
  body('branch_id').optional().isInt(),
  body('c1_message').optional(),
  body('c2_message').optional(),
  body('g1_message').optional(),
  body('g2_message').optional(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const { branch_id, c1_message, c2_message, g1_message, g2_message } = req.body;
      const message = await db.Message.create({
        branch_id: branch_id || null,
        c1_message: c1_message || null,
        c2_message: c2_message || null,
        g1_message: g1_message || null,
        g2_message: g2_message || null,
        user_id: req.userId
      });
      res.json({ success: true, message: message.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to add message' });
    }
  }
];

exports.edit = [
  body('branch_id').optional().isInt(),
  body('c1_message').optional(),
  body('c2_message').optional(),
  body('g1_message').optional(),
  body('g2_message').optional(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const message = await db.Message.findByPk(req.params.id);
      if (!message) return res.status(404).json({ success: false, message: 'Message not found' });
      if (message.user_id !== req.userId) {
        return res.status(403).json({ success: false, message: 'Forbidden' });
      }
      const { branch_id, c1_message, c2_message, g1_message, g2_message } = req.body;
      await message.update({
        branch_id: branch_id || null,
        c1_message: c1_message || null,
        c2_message: c2_message || null,
        g1_message: g1_message || null,
        g2_message: g2_message || null,
        user_id: req.userId
      });
      res.json({ success: true, message: message.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to update message' });
    }
  }
];

exports.del = async (req, res) => {
  try {
    const message = await db.Message.findByPk(req.params.id);
    if (!message) return res.status(404).json({ success: false, message: 'Message not found' });
    if (message.user_id !== req.userId) {
      return res.status(403).json({ success: false, message: 'Forbidden' });
    }
    await message.destroy();
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to delete' });
  }
};
